TITLE Windows Application

.386
.model flat, stdcall

include Win32.inc
includelib User32.lib
includelib GDI32.lib

.nolist

; constants
NULL			EQU 0

; window style
mMainWndStyle	= WS_VISIBLE+WS_CAPTION+WS_SYSMENU

; area size parameters
mGridNumX		= 12 ; <= 16
mGridNumY		= 20
mExGridNumY	= mGridNumY + 4

; drawing parameters
mLineWidth		= 1
mGridWidth		= 20
mAreaWidth		= mLineWidth * (mGridNumX + 1) + mGridWidth * mGridNumX
mAreaHeight	= mLineWidth * (mGridNumY + 1) + mGridWidth * mGridNUmY
mPanelWidth	= 200
mGameStatePosX		= mAreaWidth + 20
mGameStatePosY		= 20
mNextBlockLablePosX	= mAreaWidth + 15
mNextBlockLablePosY	= 70
mNextBlockPosX		= mAreaWidth + 50
mNextBlockPosY		= 100
mScoreInfoPosX		= mAreaWidth + 15
mScoreInfoPosY		= 210
mHelpInfoPosX			= mAreaWidth + 15
mHelpInfoPosY			= 300
mWinBgColor			= 0000ff00h
mGridColor				= 00ff0000h
mLineColor				= 00000000h

; game state
mGameIdle			= 1
mGameRunning		= 2
mGamePaused		= 3
mGameOver		= 4

; timer parameters
mTimerID			= 1
mTimerInterval		= 200

; virtual key codes
mGameStartKey		= 0Dh ; enter key
mGamePauseKey	= 20h  ; space key
mBlockLeftKey		= 25h  ; left arrow key
mBlockRightKey	= 27h  ; right arrow key
mBlockRotateKey	= 26h  ; up arrow key
mBlockDownKey	= 28h  ; down arrow key

; block
sBlock struct
	gridBits			word ?
	reserved		word ?
	minCol			byte ?
	maxCol			byte ?
	minRow		byte ?
	maxRow		byte ?
	maxRowPerCol	byte 4 dup(0)
	minColPerRow	byte 4 dup(0)
	maxColPerRow	byte 4 dup(0)
sBlock ends

; area
sArea struct
	gridBits			word  mExGridNumY dup (0) ; one line, one word
sArea ends

; function declaration
WinProc proto, hWnd:dword, localMsg:dword, wParam:dword, lParam:dword
DrawGridLines proto, hDC: dword
IsAbledMoveBlockLeft proto
IsAbledMoveBlockRight proto
IsAbledMoveBlockDown proto
PutDownBlock proto
RemoveFullGridLines proto
GenerateNewBlockID proto
InitBlockOffset proto
ClearActiveBlock proto
GetBlockPtrInSets proto
GetNextBlockPtrInSets proto
GetNextRotateBlockPtrInSets proto
GetBlockMaxYOffset proto
GenerateScoreNumStr proto
DrawGameIdleUI proto
DrawGameOverUI proto
DrawGameRunningUI proto
DrawGamePausedUI proto
GameInit proto
GameStart proto
GameStop proto
GameOver proto
GamePause proto
GameRestore proto
GetAreaGrid proto, m: dword, n: dword
GetBlockGrid proto, pBlock: ptr byte, m: dword, n: dword
IsAreaFull proto
IsAbleMoveBlockLeft proto
IsAbleMoveBlockRight proto
IsAbleMoveBlockDown proto
IsAbleRotateBlock proto

.list

.data

; window parameters
WndName	byte "Win32 ASM App",0
ClassName	byte "Win32 ASM Windows App",0
ErrorTitle	byte "Error window",0
MainWin	WNDCLASS <NULL, WinProc, NULL, NULL, NULL, NULL, NULL, COLOR_WINDOW, NULL, ClassName>
msg		MSGStruct <>
winRect	RECT <>
hMainWnd		dword ?
hInstance		dword ?
hMainDC		dword ?
hWndBgBrush	dword ?

; area and block info
BlockSets	sBlock 7*4 dup (<>) ; block sets
Area		sArea   <> ; area
BlockID			byte 0
BlockRotateID	byte 0
BlockOffsetX	byte ? ; the position of top-left corner
BlockOffsetY	byte ?
NextBlockID		byte 0
NextBlockRotateID	byte 0

; game state
GameState		dword ? ; stopped / running / paused

; score
GameScore		dword ?

; game UI strings
GameIdleStr			byte "PLAY !!!"
GamePausedStr		byte "PAUSED ???"
GameOverStr		byte "GAME OVER !!!  AGAIN???"
NextBlockLable		byte "--- Next Block ---"
ScoreInfoStr		byte "--- Score: ---"
ScoreNumStr		byte 10h dup (0)
HelpInfoTitle		byte "--- Help ---"
HelpInfoStr1		byte "Start/Stop: Enter key"
HelpInfoStr2		byte "Pause/Restore: Space key"
HelpInfoStr3		byte "Control: 4 Arrow keys"

.code

; ---------------------------------------------- entry point function
WinMain proc
	local winWidth: dword, winHeight: dword
	
	; register window class
	invoke GetModuleHandle, NULL
	mov hInstance, eax
	mov MainWin.hInstance, eax
	
	invoke LoadIcon, NULL, IDI_APPLICATION
	mov MainWin.hIcon, eax
	invoke LoadCursor, NULL, IDC_ARROW
	mov MainWin.hCursor, eax
	
	invoke RegisterClass, addr MainWin
	.if eax == 0
		call ErrorHandler
		jmp Exit_Program
	.endif
	
	; window background brush
	invoke CreateSolidBrush, mWinBgColor
	mov hWndBgBrush, eax
	
	; adjust the window size
	mov winWidth, mAreaWidth
	mov winHeight, mAreaHeight
	
	mov winRect.left, 0
	mov winRect.top, 0
	mov eax, winWidth
	dec  eax
	mov winRect.right, eax
	mov eax, winHeight
	dec  eax
	mov winRect.bottom, eax
	
	invoke AdjustWindowRectEx, addr winRect, mMainWndStyle, 0, NULL
	
	mov eax, winRect.right
	sub  eax, winRect.left
	inc    eax
	add  eax, mPanelWidth
	mov winWidth, eax
	mov eax, winRect.bottom
	sub  eax, winRect.top
	inc    eax
	mov winHeight, eax
	
	; create the window
	invoke CreateWindowEx, 0, addr ClassName, addr WndName, mMainWndStyle,
							CW_USEDEFAULT, CW_USEDEFAULT, winWidth, winHeight,
							NULL, NULL, hInstance, NULL
	.if eax == 0
		call ErrorHandler
		jmp Exit_Program
	.endif
	
	mov hMainWnd, eax
	invoke ShowWindow, hMainWnd, SW_SHOW
	invoke UpdateWindow, hMainWnd
	
	; message loop
	Message_Loop:
	
	invoke GetMessage, addr msg, NULL, NULL, NULL
	.if eax == 0
		jmp Exit_Program
	.endif
	invoke TranslateMessage, addr msg
	invoke DispatchMessage, addr msg
	jmp Message_Loop
	
	Exit_Program:
	invoke DeleteObject, hWndBgBrush
	ret
WinMain endp

; ---------------------------------------------- windows message proc function
WinProc proc uses ebx, hWnd:dword, localMsg:dword, wParam:dword, lParam:dword
	local ps: PAINTSTRUCT, brushGrid: dword, penLine: dword
	
	mov eax, localMsg
	.if eax == WM_PAINT
		invoke BeginPaint, hWnd, addr ps
		mov hMainDC, eax
		invoke SetBkMode, hMainDC, TRANSPARENT
		
		invoke CreateSolidBrush, mGridColor
		mov brushGrid, eax
		invoke SelectObject, hMainDC, brushGrid
		invoke CreatePen, PS_SOLID, 1, mLineColor
		mov penLine, eax
		invoke SelectObject, hMainDC, penLine
		
		mov eax, GameState
		.if eax == mGameIdle
			invoke DrawGameIdleUI
		.elseif eax == mGameOver
			invoke DrawGameOverUI
		.elseif eax == mGameRunning
			invoke DrawGameRunningUI
		.elseif eax == mGamePaused
			invoke DrawGamePausedUI
		.else
		.endif
		
		invoke DeleteObject, brushGrid
		invoke DeleteObject, penLine
		invoke EndPaint, hWnd, addr ps
		xor eax, eax
		jmp WinProcExit
	.elseif eax == WM_TIMER
		.if wParam != mTimerID
			xor eax, eax
			jmp WinProcExit
		.endif
		.if BlockID != -1
			invoke IsAbleMoveBlockDown
			.if eax > 0
				inc BlockOffsetY
			.else
				invoke KillTimer, hMainWnd, mTimerID
				invoke PutDownBlock
				invoke RemoveFullGridLines
				invoke ClearActiveBlock
				invoke IsAreaFull
				.if eax > 0
					invoke GameOver ; lose game
				.else
					invoke SetTimer, hMainWnd, mTimerID, mTimerInterval, NULL
				.endif
			.endif
		.else
			invoke GenerateNewBlockID
			invoke InitBlockOffset
		.endif
		invoke InvalidateRect, hMainWnd, NULL, 1
		xor eax, eax
		jmp WinProcExit
	.elseif eax == WM_KEYDOWN
		mov eax, wParam
		mov ebx, GameState
		.if eax == mGameStartKey ; game start key
			.if ebx == mGameIdle
				invoke GameStart
			.elseif ebx == mGameOver
				invoke GameStart
			.elseif ebx == mGameRunning
				invoke GameStop
			.elseif ebx == mGamePaused
				invoke GameStop
			.else
			.endif
		.elseif eax == mGamePauseKey ; game pause key
			.if ebx == mGameIdle
			.elseif ebx == mGameOver
			.elseif ebx == mGameRunning
				invoke GamePause
			.elseif ebx == mGamePaused
				invoke GameRestore
			.else
			.endif
		.elseif eax == mBlockLeftKey ; move block left key
			.if ebx == mGameIdle
			.elseif ebx == mGameOver
			.elseif ebx == mGameRunning
				invoke IsAbleMoveBlockLeft
				.if eax > 0
					dec BlockOffsetX
				.endif
			.elseif ebx == mGamePaused
			.else
			.endif
		.elseif eax == mBlockRightKey ; move block right key
			.if ebx == mGameIdle
			.elseif ebx == mGameOver
			.elseif ebx == mGameRunning
				invoke IsAbleMoveBlockRight
				.if eax > 0
					inc BlockOffsetX
				.endif
			.elseif ebx == mGamePaused
			.else
			.endif
		.elseif eax == mBlockRotateKey ; rotate block key
			.if ebx == mGameIdle
			.elseif ebx == mGameOver
			.elseif ebx == mGameRunning
				invoke IsAbleRotateBlock
				.if eax > 0
					movzx ax, BlockRotateID
					inc ax
					mov bl, 4h
					div bl
					mov BlockRotateID, ah
				.endif
			.elseif ebx == mGamePaused
			.else
			.endif
		.elseif eax == mBlockDownKey ; move block down key
			.if ebx == mGameIdle
			.elseif ebx == mGameOver
			.elseif ebx == mGameRunning
				invoke GetBlockMaxYOffset
				.if eax > 0
					push eax
					invoke KillTimer, hMainWnd, mTimerID
					pop eax
					mov BlockOffsetY, al
					invoke PutDownBlock
					invoke RemoveFullGridLines
					invoke ClearActiveBlock
					invoke IsAreaFull
					.if eax > 0
						invoke GameOver ; lose game
					.else
						invoke SetTimer, hMainWnd, mTimerID, mTimerInterval, NULL
					.endif
				.endif
			.elseif ebx == mGamePaused
			.else
			.endif
		.else
		.endif
		invoke InvalidateRect, hMainWnd, NULL, 1
		xor eax, eax
		jmp WinProcExit
	.elseif eax == WM_CREATE
		invoke SetClassLong, hWnd, GCL_HBRBACKGROUND, hWndBgBrush
		invoke GameInit ; game init here
		xor eax, eax
		jmp WinProcExit
	.elseif eax == WM_CLOSE
		invoke PostQuitMessage, 0
		xor eax, eax
		jmp WinProcExit
	.else
		invoke DefWindowProc, hWnd, localMsg, wParam, lParam
		jmp WinProcExit
	.endif
	
	WinProcExit:
	
	ret
WinProc endp

; ---------------------------------------------- error handler function
ErrorHandler proc
.data
pErrorMsg	dword ?
messageID	dword ?
.code
	invoke GetLastError
	mov messageID, eax
	invoke FormatMessage, FORMAT_MESSAGE_ALLOCATE_BUFFER + FORMAT_MESSAGE_FROM_SYSTEM,
							NULL, messageID, NULL, addr pErrorMsg, NULL, NULL
	invoke MessageBox, NULL, pErrorMsg, addr ErrorTitle, MB_ICONERROR+MB_OK
	invoke LocalFree, pErrorMsg
	ret
ErrorHandler endp

; ---------------------------------------------- draw functions
DrawGridLines proc uses eax, hDC: dword
	local sx: dword, sy: dword, ex: dword, ey: dword, cnt: dword
	
	mov cnt, mGridNumY
	inc    cnt
	mov sx, 0
	mov ex, mAreaWidth
	mov sy, 0
	mov ey, 0
		
	DrawLinesYLoop:
		
	invoke MoveToEx, hDC, sx, sy, NULL
	invoke LineTo, hDC, ex, ey
	mov eax, mLineWidth
	add eax, mGridWidth
	add sy, eax
	add ey, eax
	dec cnt
	jnz DrawLinesYLoop
		
	mov cnt, mGridNumX
	inc    cnt
	mov sy, 0
	mov ey, mAreaHeight
	mov sx, 0
	mov ex, 0
		
	DrawLinesXLoop:

	invoke MoveToEx, hDC, sx, sy, NULL
	invoke LineTo, hDC, ex, ey
	mov eax, mLineWidth
	add eax, mGridWidth
	add sx, eax
	add ex, eax
	dec cnt
	jnz DrawLinesXLoop
	ret
DrawGridLines endp

DrawSingleGrid proc, m: dword, n: dword
	pushad
	mov ecx, n
	sub ecx, 4h
	.if ecx < 0
		jmp DrawGridExit
	.endif
	mov esi, mGridWidth
	add esi, mLineWidth
	mov eax, esi
	xor edx, edx
	mov ecx, m
	mul ecx ; left over
	push eax
	mov eax, esi
	mov ecx, n
	sub ecx, 4h
	mul ecx
	mov ebx, eax ; top over
	pop eax
	mov ecx, eax
	add ecx, esi
	add ecx, mLineWidth ; right over
	mov edx, ebx
	add edx, esi
	add edx, mLineWidth ; bottom over
	invoke Rectangle, hMainDC, eax, ebx, ecx, edx
	DrawGridExit:
	popad
	ret
DrawSingleGrid endp

DrawArea proc
	pushad
	invoke DrawGridLines, hMainDC ; draw grid lines
	mov ebx, 4h
	nloop:
	xor eax, eax
	mloop:
	push eax
	invoke GetAreaGrid, eax, ebx
	.if eax > 0
		pop eax
		invoke DrawSingleGrid, eax, ebx
	.else
		pop eax
	.endif
	inc eax
	cmp eax, mGridNumX
	jb mloop
	inc ebx
	cmp ebx, mExGridNumY
	jb nloop
	popad
	ret
DrawArea endp

DrawBlock proc
	local pBlock: ptr byte
	pushad
	.if BlockID == -1
		jmp DrawBlockExit
	.endif
	invoke GetBlockPtrInSets
	mov pBlock, eax
	
	xor eax, eax
	xor ebx, ebx
	mov esi, pBlock
	mov bl, byte ptr [esi + 6] ; minRow
	mov dl, byte ptr [esi + 7] ; maxRow
	nloop:
	mov esi, pBlock
	mov al, byte ptr [esi + 4] ; minCol
	mov cl, byte ptr [esi + 5] ; maxCol
	mloop:
	push eax
	invoke GetBlockGrid, pBlock, eax, ebx
	.if eax > 0
		pop eax
		movsx esi, BlockOffsetX ; notice: the min value of BlockOffsetX is -1, so 'movsx' is used
		add esi, eax
		movsx edi, BlockOffsetY
		add edi, ebx
		invoke DrawSingleGrid, esi, edi
	.else
		pop eax
	.endif
	inc al
	cmp al, cl
	jbe mloop
	inc bl
	cmp bl, dl
	jbe nloop
	DrawBlockExit:
	popad
	ret
DrawBlock endp

DrawNextBlock proc
	local pNextBlock: ptr byte
	pushad
	invoke GetNextBlockPtrInSets
	mov pNextBlock, eax
	
	; draw next block lable
	mov eax, mNextBlockLablePosX
	mov ebx, mNextBlockLablePosY
	pushad
	invoke TextOut, hMainDC, eax, ebx, addr NextBlockLable, lengthof NextBlockLable
	popad
	
	; draw next block
	xor eax, eax
	xor ebx, ebx
	mov esi, pNextBlock
	mov bl, byte ptr [esi + 6] ; minRow
	mov dl, byte ptr [esi + 7] ; maxRow
	nloop:
	mov esi, pNextBlock
	mov al, byte ptr [esi + 4] ; minCol
	mov cl, byte ptr [esi + 5] ; maxCol
	mloop:
	push edx
	push ecx
	push ebx
	push eax
	push eax
	invoke GetBlockGrid, pNextBlock, eax, ebx
	.if eax > 0
		pop eax
		
		mov esi, mGridWidth
		add  esi, mLineWidth
		xor edx, edx
		mul esi
		push eax
		mov eax, ebx
		mul esi
		mov ebx, eax
		pop eax
		add eax, mNextBlockPosX ; eax: left over
		add ebx, mNextBlockPosY ; ebx: top over
		
		mov ecx, eax
		add ecx, esi
		add ecx, mLineWidth ; ecx: right over
		mov edx, ebx
		add edx, esi
		add edx, mLineWidth ; edx: bottom over
		invoke Rectangle, hMainDC, eax, ebx, ecx, edx
		
	.else
		pop eax
	.endif
	pop eax
	pop ebx
	pop ecx
	pop edx
	inc al
	cmp al, cl
	jbe mloop
	inc bl
	cmp bl, dl
	jbe nloop
	
	popad
	ret
DrawNextBlock endp

DrawScoreInfo proc
	pushad
	mov eax, mScoreInfoPosX
	mov ebx, mScoreInfoPosY
	invoke TextOut, hMainDC, eax, ebx, addr ScoreInfoStr, lengthof ScoreInfoStr
	popad
	pushad
	invoke GenerateScoreNumStr
	mov ebx, offset ScoreNumStr
	add ebx, 10h
	sub ebx, eax
	mov ecx, mScoreInfoPosX
	add ecx, 80
	mov edx, mScoreInfoPosY
	invoke TextOut, hMainDC, ecx, edx, eax, ebx
	popad
	ret
DrawScoreInfo endp

DrawHelpInfo proc uses ebx ecx edx
	mov ecx, mHelpInfoPosX
	mov edx, mHelpInfoPosY
	mov ebx, 20
	pushad
	invoke TextOut, hMainDC, ecx, edx, addr HelpInfoTitle, lengthof HelpInfoTitle
	popad
	add edx, ebx
	pushad
	invoke TextOut, hMainDC, ecx, edx, addr HelpInfoStr1, lengthof HelpInfoStr1
	popad
	add edx, ebx
	pushad
	invoke TextOut, hMainDC, ecx, edx, addr HelpInfoStr2, lengthof HelpInfoStr2
	popad
	add edx, ebx
	pushad
	invoke TextOut, hMainDC, ecx, edx, addr HelpInfoStr3, lengthof HelpInfoStr3
	popad
	ret
DrawHelpInfo endp

DrawGameIdleUI proc
	pushad
	invoke DrawArea
	mov ecx, mGameStatePosX
	mov edx, mGameStatePosY
	invoke TextOut, hMainDC, ecx, edx, addr GameIdleStr, lengthof GameIdleStr
	popad
	invoke DrawHelpInfo
	ret
DrawGameIdleUI endp

DrawGameRunningUI proc
	pushad
	invoke DrawArea
	invoke DrawBlock
	invoke DrawNextBlock
	invoke DrawScoreInfo
	invoke DrawHelpInfo
	popad
	ret
DrawGameRunningUI endp

DrawGamePausedUI proc
	invoke DrawArea
	invoke DrawBlock
	invoke DrawNextBlock
	pushad
	mov ecx, mGameStatePosX
	mov edx, mGameStatePosY
	invoke TextOut, hMainDC, ecx, edx, addr GamePausedStr, lengthof GamePausedStr
	popad
	invoke DrawScoreInfo
	invoke DrawHelpInfo
	ret
DrawGamePausedUI endp

DrawGameOverUI proc
	invoke DrawArea
	pushad
	mov ecx, mGameStatePosX
	mov edx, mGameStatePosY
	invoke TextOut, hMainDC, ecx, edx, addr GameOverStr, lengthof GameOverStr
	popad
	invoke DrawScoreInfo
	invoke DrawHelpInfo
	ret
DrawGameOverUI endp

; ---------------------------------------------- block operate function
ClearAllBlockGrids proc, pBlock: ptr byte
	push esi
	mov esi, pBlock
	mov word ptr [esi + 0], 0h
	pop esi
	ret
ClearAllBlockGrids endp

SetBlockGrid proc uses eax ebx ecx edi, pBlock: ptr byte, m: dword, n: dword, bit: dword
	mov edi, pBlock
	mov eax, m
	mov cl, 04h
	mul  cl
	add  eax, n
	mov cl, al
	.if bit > 0
		mov bx, 01h
		shl    bx, cl
		or     word ptr [edi + 0], bx
	.else
		mov bx, 0fffeh
		rol    bx, cl
		and  word ptr [edi +0], bx
	.endif
	ret
SetBlockGrid endp

GetBlockGrid proc uses ebx ecx edx esi, pBlock: ptr byte, m: dword, n: dword
	mov esi, pBlock
	mov eax, m
	mov cl, 04h
	mul cl
	add eax, n
	mov bx, 01h
	mov cl, al
	shl    bx, cl
	xor edx, edx
	mov dx, word ptr [esi +0]
	and dx, bx
	shr  dx, cl
	mov eax, edx
	ret
GetBlockGrid endp

BuildBlockInfo proc, pBlock: ptr byte
	local minCol: byte, maxCol: byte, minRow: byte, maxRow: byte
	pushad

	; calculate minCol, maxCol, minColPerRow[] and maxColPerRow[]
	mov minCol, 03h
	mov maxCol, 0h
	
	mov edx, 04h      ; n counter
	xor   ebx, ebx      ; n
	
	nloop:
	
	mov ecx, 04h      ; m counter
	xor   eax, eax       ; m
	mov esi, 03h	; minColPerRow
	xor   edi, edi         ; maxColPerRow
	
	mloop:
	
	push eax
	invoke GetBlockGrid, pBlock, eax, ebx
	.if eax > 0
		pop eax
		.if eax < esi
			mov esi, eax
		.endif
		.if eax > edi
			mov edi, eax
		.endif
	.else
		pop eax
	.endif
	inc eax
	loopnz mloop
	
	push eax
	push edx
	mov ax, si
	mov edx, pBlock
	add edx, 12
	mov byte ptr [edx + ebx], al ; minColPerRow[]	
	.if al < minCol
		mov minCol, al
	.endif
	mov ax, di
	add edx, 4
	mov byte ptr [edx + ebx], al ; maxColPerRow[]
	.if al > maxCol
		mov maxCol, al
	.endif
	pop edx
	pop eax
	
	inc   ebx
	dec  edx

	jnz nloop
	
	mov edx, pBlock
	mov al, minCol
	mov byte ptr [edx + 4], al ; minCol
	mov al, maxCol
	mov byte ptr [edx + 5], al ; maxCol
	
	; calculate minRow, maxRow and maxRowPerCol[]
	mov minRow, 03h
	mov maxRow, 0h
		
	mov edx, 04h      ; m counter
	xor   ebx, ebx      ; m
		
	mloop2:
		
	mov ecx, 04h      ; n counter
	xor   eax, eax       ; n
	mov esi, 03h	; minRowPerCol
	xor   edi, edi         ; maxRowPerCol
		
	nloop2:
		
	push eax
	invoke GetBlockGrid, pBlock, ebx, eax
	.if eax > 0
		pop eax
		.if eax < esi
			mov esi, eax
		.endif
		.if eax > edi
			mov edi, eax
		.endif
	.else
		pop eax
	.endif
	inc eax
	loopnz nloop2
		
	push eax
	push edx
	mov ax, si
	.if al < minRow
		mov minRow, al
	.endif
	mov ax, di
	mov edx, pBlock
	add edx, 8
	mov byte ptr [edx + ebx], al ; maxRowPerCol[]
	.if al > maxRow
		mov maxRow, al
	.endif
	pop edx
	pop eax
		
	inc   ebx
	dec  edx
	
	jnz mloop2
	
	mov edx, pBlock
	mov al, minRow
	mov byte ptr [edx + 6], al ; minRow
	mov al, maxRow
	mov byte ptr [edx + 7], al ; maxRow
	
	popad
	ret
BuildBlockInfo endp

; ---------------------------------------------- area operate function
ClearAllAreaGrids proc
	pushad
	mov ecx, lengthof Area.gridBits
	mov edi, offset Area.gridBits
	xor ax, ax
	cld
	rep stosw
	popad
	ret
ClearAllAreaGrids endp

IsAreaFull proc uses ebx ecx esi
	mov esi, offset Area.gridBits
	mov ax, word ptr [esi + 4h]
	mov cl, 10h
	sub cl, mGridNumX
	mov bx, 0ffffh
	shr bx, cl
	and ax, bx
	or ax, ax
	jnz FullExit
	xor eax, eax
	ret
	FullExit:
	mov eax, 01h
	ret
IsAreaFull endp

SetAreaGrid proc, m: dword, n: dword, bit: dword
	pushad
	mov edi, offset Area.gridBits
	mov eax, type Area.gridBits
	mul n
	add edi, eax
	mov ecx, m
	.if bit > 0
		mov bx, 01h
		shl   bx, cl
		or     word ptr [edi], bx
	.else
		mov bx, 0fffeh
		rol    bx, cl
		and  word ptr [edi], bx
	.endif
	popad
	ret
SetAreaGrid endp

GetAreaGrid proc uses ebx ecx edx esi, m: dword, n: dword
	mov esi, offset Area.gridBits
	mov eax, type Area.gridBits
	mul n
	add esi, eax
	mov ecx, m
	mov bx, 1h
	shl    bx, cl
	mov dx, word ptr [esi]
	and  dx, bx
	shr   dx, cl
	movzx eax, dx
	ret
GetAreaGrid endp

; ---------------------------------------------- logic control
IsAbleMoveBlockLeft proc uses ebx ecx edx esi
	local pBlock: ptr byte
	.if BlockID == -1
		jmp NotAbleExit
	.endif
	invoke GetBlockPtrInSets
	mov pBlock, eax
	mov esi, pBlock
	xor eax, eax
	xor ebx, ebx
	xor ecx, ecx
	mov al, BlockOffsetX
	add  al, byte ptr [esi + 4] ; + minCol
	or al, al
	jz NotAbleExit
	mov cl, byte ptr [esi + 6] ; minRow
	mov dl, byte ptr [esi + 7] ; maxRow
	rowLoop:
	mov al, BlockOffsetX
	mov bl, BlockOffsetY
	add  al, byte ptr [esi + 12 + ecx] ; + minColPerRow
	add  bl, cl
	dec  al
	invoke GetAreaGrid, eax, ebx
	.if eax > 0
		jmp NotAbleExit
	.endif
	inc cl
	cmp cl, dl
	jbe rowLoop
	mov eax, 01h
	ret
	NotAbleExit:
	xor eax, eax
	ret
IsAbleMoveBlockLeft endp

IsAbleMoveBlockRight proc uses ebx ecx edx esi
	local pBlock: ptr byte
	.if BlockID == -1
		jmp NotAbleExit
	.endif
	invoke GetBlockPtrInSets
	mov pBlock, eax
	mov esi, pBlock
	xor eax, eax
	xor ebx, ebx
	xor ecx, ecx
	mov al, BlockOffsetX
	add  al, byte ptr [esi + 5] ; + maxCol
	mov bl, mGridNumX
	dec  bl
	cmp al, bl
	jae NotAbleExit
	mov cl, byte ptr [esi + 6] ; minRow
	mov dl, byte ptr [esi + 7] ; maxRow
	rowLoop:
	mov al, BlockOffsetX
	mov bl, BlockOffsetY
	add  al, byte ptr [esi + 16 + ecx] ; + maxCloPerRow
	add  bl, cl
	inc  al
	invoke GetAreaGrid, eax, ebx
	.if eax > 0
		jmp NotAbleExit
	.endif
	inc cl
	cmp cl, dl
	jbe rowLoop
	mov eax, 01h
	ret
	NotAbleExit:
	xor eax, eax
	ret
IsAbleMoveBlockRight endp

IsAbleMoveBlockDown proc uses ebx ecx edx esi
	local pBlock: ptr byte
	.if BlockID == -1
		jmp NotAbleExit
	.endif
	invoke GetBlockPtrInSets
	mov pBlock, eax
	mov esi, pBlock
	xor eax, eax
	xor ebx, ebx
	xor ecx, ecx
	mov bl, BlockOffsetY
	add  bl, byte ptr [esi + 7] ; + maxRow
	mov al, mExGridNumY
	dec  al
	cmp bl, al
	jae NotAbleExit
	mov cl, byte ptr [esi + 4] ; minCol
	mov dl, byte ptr [esi + 5] ; maxCol
	colLoop:
	mov al, BlockOffsetX
	mov bl, BlockOffsetY
	add  al, cl
	add  bl, byte ptr [esi + 8 + ecx] ; + maxRowPerCol
	inc  bl
	invoke GetAreaGrid, eax, ebx
	.if eax > 0
		jmp NotAbleExit
	.endif
	inc cl
	cmp cl, dl
	jbe colLoop
	mov eax, 01h
	ret
	NotAbleExit:
	xor eax, eax
	ret
IsAbleMoveBlockDown endp

IsAbleRotateBlock proc uses esi
	local pNextBlock: ptr byte
	.if BlockID == -1
		jmp NotAbleExit
	.endif
	invoke GetNextRotateBlockPtrInSets
	mov pNextBlock, eax
	mov esi, pNextBlock
	
	xor eax, eax
	xor ebx, ebx
	xor ecx, ecx
	xor edx, edx
	mov cl, byte ptr [esi + 6] ; minRow
	mov dl, byte ptr [esi + 7] ; maxRow
	nloop:
	mov al, byte ptr [esi + 12 + ecx] ; minColPerRow
	mov bl, byte ptr [esi + 16 + ecx] ; maxColPerRow
	mloop:
	push eax
	invoke GetBlockGrid, pNextBlock, eax, ecx
	.if eax > 0
		pop eax
		push ebx
		push edx
		xor ebx, ebx
		xor edx, edx
		mov bl, BlockOffsetX
		mov dl, BlockOffsetY
		add  bl, al
		add  dl, cl
		.if bl < 0 ; beyond the left border
			pop edx ; be careful here
			pop ebx
			jmp NotAbleExit
		.endif
		.if bl >= mGridNumX-1 ; beyond the right border
			pop edx ; be careful here
			pop ebx
			jmp NotAbleExit
		.endif
		.if dl >= mExGridNumY-1 ; beyond the bottom border
			pop edx ; be careful here
			pop ebx
			jmp NotAbleExit
		.endif
		push eax
		invoke GetAreaGrid, ebx, edx
		.if eax > 0 ; grid covered
			pop eax
			pop edx ; be careful here
			pop ebx
			jmp NotAbleExit
		.else
			pop eax
		.endif
		pop edx
		pop ebx
	.else
		pop eax
	.endif
	inc al
	cmp al, bl
	jbe mloop
	inc cl
	cmp cl, dl
	jbe nloop
	
	mov eax, 01h
	ret
	NotAbleExit:
	xor eax, eax
	ret
IsAbleRotateBlock endp

PutDownBlock proc
	local pBlock: ptr byte
	pushad
	invoke GetBlockPtrInSets
	mov pBlock, eax
	mov esi, pBlock
	xor eax, eax
	xor ebx, ebx
	xor ecx, ecx
	xor edx, edx
	mov cl, byte ptr [esi + 6] ; minRow
	mov dl, byte ptr [esi + 7] ; maxRow
	nloop:
	mov al, byte ptr [esi + 12 + ecx] ; minColPerRow
	mov bl, byte ptr [esi + 16 + ecx] ; maxColPerRow
	mloop:
	push eax
	invoke GetBlockGrid, pBlock, eax, ecx
	.if eax > 0
		pop eax
		push ebx
		push edx
		xor ebx, ebx
		xor edx, edx
		mov bl, BlockOffsetX
		mov dl, BlockOffsetY
		add  bl, al
		add  dl, cl
		.if dl >= 0 ; too high in case
			invoke SetAreaGrid, ebx, edx, 01h
		.endif
		pop edx
		pop ebx
	.else
		pop eax
	.endif
	inc al
	cmp al, bl
	jbe mloop
	inc cl
	cmp cl, dl
	jbe nloop
	popad
	ret
PutDownBlock endp

RemoveFullGridLines proc uses ebx ecx edx esi edi ; return the number of lines removed
	local pBlock: ptr byte, lineCnt: dword, lineBegin: dword, lineEnd: dword
	invoke GetBlockPtrInSets
	mov pBlock, eax
	mov esi, pBlock
	mov lineCnt, 0h
	
	movzx eax, byte ptr [esi + 7] ; maxRow of Block
	add al, BlockOffsetY
	mov lineBegin, eax
	mov lineEnd, 4h
	
	rowLoop:
	
	mov esi, offset Area.gridBits
	mov eax, type Area.gridBits
	mov edx, lineBegin
	mul dl
	add esi, eax
	mov bx, word ptr [esi] ; the grid line to check
	mov dx, 0ffffh
	mov cl, 10h
	sub cl, mGridNumX
	shr dx, cl
	and bx, dx
	cmp bx, dx
	je FullGridLine
	
	.if lineCnt > 0
		mov edi, esi
		mov eax, type Area.gridBits
		mov edx, lineCnt
		mul dl
		add edi, eax
		mov word ptr [edi], bx
	.endif
	jmp CheckOver
	
	FullGridLine:
	inc lineCnt
	
	CheckOver:
	dec lineBegin
	mov ecx, lineBegin
	cmp ecx, lineEnd
	jae rowLoop
	
	.if lineCnt > 0
		mov ecx, lineCnt
		mov edi, offset Area.gridBits
		mov eax, type Area.gridBits
		mov edx, lineEnd
		mul dl
		add edi, eax
		xor ax, ax
		cld
		rep stosw
		; calculate the score
		.if lineCnt == 1
			add GameScore, 10
		.elseif lineCnt == 2
			add GameScore, 30
		.elseif lineCnt == 3
			add GameScore, 70
		.elseif lineCnt == 4
			add GameScore, 130
		.else
		.endif
	.endif
	
	mov eax, lineCnt
	ret
RemoveFullGridLines endp

InitBlockOffset proc uses eax
	mov eax, mGridNumX
	shr eax, 1h
	sub eax, 2h
	mov BlockOffsetX, al
	mov BlockOffsetY, 0h
	ret
InitBlockOffset endp

GenerateNewBlockID proc
	pushad
	; mov next ID to current ID
	mov al, NextBlockID
	mov bl, NextBlockRotateID
	mov BlockID, al
	mov BlockRotateID, bl
	; generate next ID
	invoke GetTickCount
	xor edx, edx
	mov ecx, 07h
	div ecx
	mov NextBlockID, dl
	xor edx, edx
	mov ecx, 04h
	div ecx
	mov NextBlockRotateID, dl
	popad
	ret
GenerateNewBlockID endp

ClearActiveBlock proc
	mov BlockID, -1
	ret
ClearActiveBlock endp

; ---------------------------------------------- game control
GameInit proc
	pushad

	mov GameState, mGameIdle
	invoke ClearAllAreaGrids
	
	; init 7*4 block sets
	mov edi, offset BlockSets
	mov ebx, type BlockSets
	xor eax, eax
	
	mov word ptr [edi + eax], 0010001000100010b ; ####
	add eax, ebx
	mov word ptr [edi + eax], 0000000011110000b
	add eax, ebx
	mov word ptr [edi + eax], 0010001000100010b
	add eax, ebx
	mov word ptr [edi + eax], 0000000011110000b
	
	add eax, ebx
	mov word ptr [edi + eax], 0000011001100000b ; ##
	add eax, ebx
	mov word ptr [edi + eax], 0000011001100000b ; ##
	add eax, ebx
	mov word ptr [edi + eax], 0000011001100000b
	add eax, ebx
	mov word ptr [edi + eax], 0000011001100000b
	
	add eax, ebx
	mov word ptr [edi + eax], 0000001001100100b ;    ##
	add eax, ebx
	mov word ptr [edi + eax], 0000011000110000b ; ##
	add eax, ebx
	mov word ptr [edi + eax], 0000001001100100b
	add eax, ebx
	mov word ptr [edi + eax], 0000011000110000b
	
	add eax, ebx
	mov word ptr [edi + eax], 0000010001100010b ; ##
	add eax, ebx
	mov word ptr [edi + eax], 0000001101100000b ;    ##
	add eax, ebx
	mov word ptr [edi + eax], 0000010001100010b
	add eax, ebx
	mov word ptr [edi + eax], 0000001101100000b
	
	add eax, ebx
	mov word ptr [edi + eax], 0000011000100010b ; ###
	add eax, ebx
	mov word ptr [edi + eax], 0000011101000000b ;     #
	add eax, ebx
	mov word ptr [edi + eax], 0000010001000110b
	add eax, ebx
	mov word ptr [edi + eax], 0000000101110000b
	
	add eax, ebx
	mov word ptr [edi + eax], 0000001000100110b ; ###
	add eax, ebx
	mov word ptr [edi + eax], 0000011100010000b ; #
	add eax, ebx
	mov word ptr [edi + eax], 0000011001000100b
	add eax, ebx
	mov word ptr [edi + eax], 0000010001110000b
	
	add eax, ebx
	mov word ptr [edi + eax], 0000001001100010b ; ###
	add eax, ebx
	mov word ptr [edi + eax], 0000011100100000b ;   #
	add eax, ebx
	mov word ptr [edi + eax], 0000010001100100b
	add eax, ebx
	mov word ptr [edi + eax], 0000001001110000b
	
	mov edi, offset BlockSets
	mov ecx, lengthof BlockSets
	setsLoop:
	invoke BuildBlockInfo, edi	
	add edi, type BlockSets
	loopnz setsLoop

	mov BlockID, -1
	mov BlockRotateID, 0
	mov BlockOffsetX, 0
	mov BlockOffsetY, 0

	popad
	ret
GameInit endp

GameStart proc
	pushad
	invoke ClearAllAreaGrids
	invoke GenerateNewBlockID
	invoke GenerateNewBlockID
	invoke InitBlockOffset
	invoke SetTimer, hMainWnd, mTimerID, mTimerInterval, NULL
	mov GameState, mGameRunning
	mov GameScore, 0h
	popad
	ret
GameStart endp

GameStop proc
	pushad
	invoke KillTimer, hMainWnd, mTimerID
	invoke ClearActiveBlock
	mov GameState, mGameIdle
	popad
	ret
GameStop endp

GameOver proc
	pushad
	invoke KillTimer, hMainWnd, mTimerID
	invoke ClearActiveBlock
	mov GameState, mGameOver
	popad
	ret
GameOver endp

GamePause proc
	pushad
	invoke KillTimer, hMainWnd, mTImerID
	mov GameState, mGamePaused
	popad
	ret
GamePause endp

GameRestore proc
	pushad
	invoke SetTimer, hMainWnd, mTImerID, mTimerInterval, NULL
	mov GameState, mGameRunning
	popad
	ret
GameRestore endp

; ---------------------------------------------- util functins
GetBlockPtrInSets proc uses ebx ecx edx esi
	movzx eax, BlockID
	xor edx, edx
	mov ecx, 4h
	mul ecx
	movzx ebx, BlockRotateID
	add eax, ebx
	mov ecx, type BlockSets
	mul ecx
	mov esi, offset BlockSets
	add esi, eax
	mov eax, esi
	ret
GetBlockPtrInSets endp

GetNextBlockPtrInSets proc uses ebx ecx edx esi
	movzx eax, NextBlockID
	xor edx, edx
	mov ecx, 4h
	mul ecx
	movzx ebx, NextBlockRotateID
	add eax, ebx
	mov ecx, type BlockSets
	mul ecx
	mov esi, offset BlockSets
	add esi, eax
	mov eax, esi
	ret
GetNextBlockPtrInSets endp

GetNextRotateBlockPtrInSets proc uses ebx ecx edx esi
	movzx eax, BlockRotateID
	inc ax
	mov bl, 4h
	div bl
	movzx ebx, ah ; get next BlockRotateID
	movzx eax, BlockID
	xor edx, edx
	mov ecx, 4h
	mul ecx
	add eax, ebx
	mov ecx, type BlockSets
	mul ecx
	mov esi, offset BlockSets
	add esi, eax
	mov eax, esi
	ret
GetNextRotateBlockPtrInSets endp

GetBlockMaxYOffset proc uses ebx ecx edx esi ; if failed, return 0, else > 0
	local pBlock: ptr byte
	.if BlockID == -1
		jmp GetFailedExit
	.endif
	invoke GetBlockPtrInSets
	mov pBlock, eax
	mov esi, pBlock
	
	mov eax, mExGridNumY
	dec eax
	sub al, byte ptr [esi + 7] ; maxYOffset
	movzx ebx, byte ptr [esi + 4] ; minCol
	mov cl, byte ptr [esi + 5]      ; maxCol
	
	colLoop:
	push ecx
	movzx ecx,  BlockOffsetX
	add cl, bl
	movzx edx, BlockOffsetY
	add dl, byte ptr [esi + 8 + ebx] ; maxRowOfPerCol[]
	
	rowLoop:
	cmp edx, mExGridNumY-1
	jae next
	
	inc edx
	push eax
	invoke GetAreaGrid, ecx, edx
	.if eax == 0
		pop eax
		jmp rowLoop
	.else
		pop eax
	.endif
	dec edx
	
	next:
	sub dl, byte ptr [esi + 8 + ebx]
	.if edx < eax
		mov eax, edx
	.endif
	
	pop ecx
	inc bl
	cmp bl, cl
	jbe colLoop
	ret
	
	GetFailedExit:
	xor eax, eax
	ret
GetBlockMaxYOffset endp

GenerateScoreNumStr proc uses ebx edx edi ; return the offset of valid digital string
	mov edi, offset ScoreNumStr
	add edi, 0fh
	mov eax, GameScore
	mov ebx, 0ah
	
	next:
	xor edx, edx
	div ebx
	add dl, '0'
	mov byte ptr [edi], dl
	dec edi
	or eax, eax
	jnz next
	
	inc edi
	mov eax, edi
	ret
GenerateScoreNumStr endp

END WinMain