/*++

Module Name :

	ListSSDT.c

Abstract:

	

--*/


#include "ListSSDT.h"


//
//	Globals
//

ULONG g_NumOfServ = 0;	//	number of services
PULONG g_ServTable = 0;	//	point to KiServiceTable


NTSTATUS 
GetNeedLength(
	OUT PULONG NameLength,
	OUT PULONG AddrLength)
/*++

Routine Description:

	get the size of all information.

Arguments:

	NameLength - return the size of name buffer. every function
		needs 0x32 bytes memory.
	AddrLength - return the size of address buffer.

Return Value:

	NT Status code

--*/
{
	PSSDT_ENTRY SSDT;
	UNICODE_STRING SSDTName;

	if( g_NumOfServ != 0 )
	{
		*NameLength = g_NumOfServ * 0x32;
		*AddrLength = g_NumOfServ * 0x04;
		return STATUS_SUCCESS;
	}

	//
	//	get the address of SSDT
	//

	RtlInitUnicodeString( &SSDTName, L"KeServiceDescriptorTable" );

	SSDT = MmGetSystemRoutineAddress( &SSDTName );

	if(SSDT == NULL)
	{
		return STATUS_UNSUCCESSFUL;
	}

	//
	//	set globals
	//

	g_NumOfServ = SSDT->NumOfServ;
	g_ServTable = SSDT->SSDTBase;

	//
	//	I'm lazy :)
	//

	return GetNeedLength( NameLength, AddrLength);
}


NTSTATUS GetSSDTInfo(
	OUT PCHAR FuncName,
	OUT PULONG FuncAddr,
	OUT PULONG AddrOrig)
/*++

Routine Description:

	main routine of ListSSDT.c
	return information of SSDT

Arguments:

	FuncName - filled with function names of SSDT.
	FuncAddr - filled with current function addresses of SSDT.
	AddrOrig - filled with original function addresses of SSDT.

Return Value:
	
	NT Status code.

Step:
	
	1. load ntdll.dll into memory, look over the original export table
		to find SSDT funcitons, then get their names.
	2. enum loaded modules in system. the first module is nt. get nt's
		base and path.
	3. load nt file into memory, look over the original export table to
		find KeSeviceDescriptorTable's offset, and then look over the
		relocation table to find KiServiceTable's offset.
	4. relocate with nt base.

--*/
{
	NTSTATUS Status;
	ULONG NameLength;
	ULONG AddrLength;
	PVOID FileBuffer;
	WCHAR NtPath[0x100];
	ULONG KrnlBase;
	ULONG i;

	//
	//	Er.. most time ,the system device is c:
	//	of cource , you can use "\SystemRoot\System32\"
	//

	CHAR NtName[0x100] = "\\??\\c:";


	//
	//	fill buffer with zero.
	//

	Status = GetNeedLength(&NameLength, &AddrLength);

	RtlZeroMemory( FuncName, NameLength);
	RtlZeroMemory( FuncAddr, AddrLength);
	RtlZeroMemory( AddrOrig, AddrLength);


	//
	//	Load ntdll.dll into memory
	//

	Status = LoadFile( L"\\??\\c:\\windows\\system32\\ntdll.dll", &FileBuffer );
	
	if(!NT_SUCCESS(Status))
	{
		return Status;
	}

	//
	//	Get function Names.
	//

	Status = GetFuncName( FileBuffer, FuncName);


	//
	//	buffer is allocated in LoadFile, but we must
	//	free it in current function
	//

	if(FileBuffer != NULL)
	{
		ExFreePool(FileBuffer);
		FileBuffer = NULL;
	}

	if(!NT_SUCCESS(Status))
	{
		return Status;
	}
	
	//
	//	get the information of first module.
	//

	Status = GetModuleByIndex( 0x00, NtName ,&KrnlBase);
	
	if(!NT_SUCCESS(Status))
	{
		return Status;
	}

	
	//
	//	the module path is Ansi, conver it to Unicode
	//

	AnsiToUnicode( NtPath, NtName);
	

	//
	//	Load nt file into memory.
	//

	Status = LoadFile( NtPath, &FileBuffer );

	if(!NT_SUCCESS(Status))
	{
		return Status;
	}

	
	//
	//	get the original funciton address of SSDT.
	//

	Status = GetFuncAddr( FileBuffer, AddrOrig);


	//
	//	free buffer
	//

	if(FileBuffer != NULL)
	{
		ExFreePool(FileBuffer);
		FileBuffer = NULL;
	}
	

	//
	// relocate
	//

	if(NT_SUCCESS(Status))
	{
		for( i = 0; i < g_NumOfServ; i++)
		{
			*(AddrOrig + i) += KrnlBase;
		}
	}

	
	//
	//	get the current function address of SSDT.
	//

	RtlCopyMemory( FuncAddr, g_ServTable, g_NumOfServ * 0x04);

	return Status;
}



NTSTATUS
LoadFile(
	IN PWCH FilePath,
	OUT PVOID * FileBuffer)
/*++

Routine Description:

	this routine load file into memory.

Arguments:

	FilePath - the path of file to be loaded.
	FileBuffer - pointer to buffer of file.

Return Value:

	NT Status code

Notice:

	the buffer is allocated in this routine ,but should
	be free in the other routine.

--*/
{
	NTSTATUS Status;
	UNICODE_STRING ImagePath;
	OBJECT_ATTRIBUTES ObjectAttributes;
	IO_STATUS_BLOCK IoStatusBlock;
	HANDLE hFile;
	FILE_STANDARD_INFORMATION FileSize;
	PVOID Buffer;

	//
	//	create file
	//

	RtlInitUnicodeString(&ImagePath, FilePath);

	InitializeObjectAttributes(&ObjectAttributes,
		&ImagePath,
		OBJ_CASE_INSENSITIVE,
		NULL,
		NULL);

	Status = ZwCreateFile( &hFile,
						   GENERIC_READ,
						   &ObjectAttributes,
						   &IoStatusBlock,
						   NULL,
						   FILE_ATTRIBUTE_NORMAL,
						   FILE_SHARE_READ,
						   FILE_OPEN,
						   FILE_SYNCHRONOUS_IO_NONALERT,
						   NULL,
						   0);

	if(!NT_SUCCESS(Status))
	{
		return Status;
	}

	do
	{
		//
		//	get file size
		//

		Status = ZwQueryInformationFile( hFile,
										 &IoStatusBlock,
										 &FileSize,
										 sizeof(FileSize),
										 FileStandardInformation);

		if(!NT_SUCCESS(Status))
		{
			break;
		}
		
		//
		//	allocate buffer
		//

		Buffer = ExAllocatePoolWithTag(	NonPagedPool, 
										FileSize.AllocationSize.QuadPart, 
										'Dll ');

		if(Buffer == NULL)
		{
			Status = STATUS_UNSUCCESSFUL;
			break;
		}

		//
		//	read file
		//

		Status = ZwReadFile( hFile,
							 NULL,
							 NULL,
							 NULL,
							 &IoStatusBlock,
							 Buffer,
							 (LONG)FileSize.AllocationSize.QuadPart,
							 NULL,
							 NULL);

		//
		//	if faild, free the buffer
		//

		if(!NT_SUCCESS(Status))
		{
			ExFreePool(Buffer);
			Buffer = NULL;
		}

	}while(FALSE);

	if(hFile != NULL)
	{
		ZwClose(hFile);
		hFile = NULL;
	}

	*FileBuffer = Buffer;

	return Status;
}


NTSTATUS
GetFuncName(
	IN PVOID FileBuffer,
	OUT PCHAR FuncName)
/*++

Routine Description:

	look over the export table, find the Zw******* functions

Arguments:

	FileBuffer - the buffer filled with ntdll.dll
	FuncName - buffer to receive function name

Return Value:

	NT Status code

--*/
{
	NTSTATUS Status;
	PIMAGE_EXPORT_DIRECTORY pImageExportDir;
	ULONG ExportSize;
	ULONG NumberOfNames;
	PUSHORT AddressOfNameOrdinals;
	PULONG AddressOfNames;
	PULONG AddressOfFunctions;
	ULONG MappedExpBase;
	ULONG Index;
	ULONG Temp;
	PUCHAR Address;
	ULONG i;

	//
	//	get the export table offset in file
	//

	pImageExportDir = RtlImageDirectoryEntryToData( FileBuffer,
													FALSE,
													IMAGE_DIRECTORY_ENTRY_EXPORT,
													&ExportSize);


	//
	//	get the export table offset in memory
	//

	MappedExpBase = (ULONG)RtlImageDirectoryEntryToData( FileBuffer,
												  TRUE,
												  IMAGE_DIRECTORY_ENTRY_EXPORT,
												  &ExportSize);

	if(pImageExportDir == NULL || MappedExpBase == 0)
	{
		return STATUS_UNSUCCESSFUL;
	}


	//
	//	find functions with special bytes
	//

	NumberOfNames = pImageExportDir->NumberOfNames;
	AddressOfNameOrdinals = (PUSHORT)(pImageExportDir->AddressOfNameOrdinals + (ULONG)FileBuffer - MappedExpBase + (ULONG)pImageExportDir);
	AddressOfNames = (PULONG)(pImageExportDir->AddressOfNames + (ULONG)FileBuffer - MappedExpBase + (ULONG)pImageExportDir);
	AddressOfFunctions = (PULONG)(pImageExportDir->AddressOfFunctions + (ULONG)FileBuffer - MappedExpBase + (ULONG)pImageExportDir);

	for( i = 0; i < NumberOfNames; i++)
	{
		Index = *(AddressOfNameOrdinals + i);
		Address = (PUCHAR)*(AddressOfFunctions + Index);
		Address = Address + (ULONG)FileBuffer - MappedExpBase + (ULONG)pImageExportDir; 

		//
		//	b8 xx xx xx xx   mov    eax, XXXX
		//	ba 00 03 fe 7f   mov    edx, 7FFE0300h
		//	ff 12            call   dword ptr [edx]
		//

		if(*Address = 0xB8 &&
		   *(Address + 0x05) == 0xBA &&
		   *(PULONG)(Address + 0x06) == 0x7FFE0300 &&
		   *(PUSHORT)(Address + 0x0A) == 0x12FF)
		{
			Temp = *(PULONG)(Address + 1);
			Address = (PUCHAR)*(AddressOfNames + Index);
			Address = Address + (ULONG)FileBuffer - MappedExpBase + (ULONG)pImageExportDir;


			//
			//	safer than strcpy
			//

			RtlCopyMemory(FuncName + Temp * 0x32, Address, 0x32);

			*(FuncName + Temp * 0x32 + 0x31) = 0x00;
		}
	}

	return STATUS_SUCCESS;
}

NTSTATUS
GetFuncAddr(
	IN PVOID FileBuffer,
	OUT PULONG FuncAddr)
/*++

Routine Description:
	
	find the KiServiceTable in file and return it.

Arguments:
	
	FileBuffer - buffer filled with nt file.
	FuncAddr - buffer to receive function address.

Return Value:

	NT Status code

--*/
{
	NTSTATUS Status;
	ULONG SSDTOffset;
	ULONG ImageBase;
	ULONG KiServAddr;
	ULONG i;

	Status = STATUS_SUCCESS;	

	
	//
	// get KeServiceDescriptorTable's offset from file
	//

	SSDTOffset = GetExportData( FileBuffer, "KeServiceDescriptorTable");

	if( SSDTOffset == 0)
	{
		return STATUS_UNSUCCESSFUL;
	}
	
	//
	//	get image base from file
	//

	ImageBase = GetImageBase( FileBuffer );

	if(ImageBase == 0)
	{
		return STATUS_UNSUCCESSFUL;
	}

	//
	//	get KiServiceTable's offset
	//

	KiServAddr = GetServiceTableAddress( FileBuffer, ImageBase + SSDTOffset);

	if(KiServAddr == 0)
	{
		return STATUS_UNSUCCESSFUL;
	}

	//
	// get function address
	//

	RtlCopyMemory(FuncAddr, (PVOID)(KiServAddr + (ULONG)FileBuffer - ImageBase), g_NumOfServ * 0x04);

	
	//
	//	relocate
	//

	for( i = 0; i < g_NumOfServ; i++)
	{
		*(FuncAddr + i) -= ImageBase;
	}

	return Status;
}

NTSTATUS
GetModuleByIndex(
	IN ULONG Index,
	OUT PCHAR ModuleName,
	OUT PULONG ModuleBase)
/*++

Routine Description:

	get loaded module's path and base

Arguments:

	Index - index of module. the 0x00 is nt
	ModuleName - return module path
	ModuleBase - return module base

Return Value:
	
	NT Status code.

--*/
{
	NTSTATUS Status;
	PVOID Buffer;
	ULONG BufferSize;
	ULONG Count;
	PSYSTEM_MODULE_INFORMATION_ENTRY pModuleInfo;
	ULONG i;

	//
	//	the second argument is null, the function
	//	return the size of buffer
	//

	Status = ZwQuerySystemInformation( 0x0b,
									   NULL,
									   0,
									   &BufferSize);

	if(!NT_SUCCESS(Status) && Status != STATUS_INFO_LENGTH_MISMATCH)
	{
		return Status;
	}

	Buffer = ExAllocatePoolWithTag( NonPagedPool, BufferSize, 'Ddk ');

	if(Buffer == NULL)
	{
		return STATUS_UNSUCCESSFUL;
	}

	Status = ZwQuerySystemInformation( 0x0b,
									   Buffer,
									   BufferSize,
									   NULL);

	if(!NT_SUCCESS(Status))
	{
		ExFreePool(Buffer);
		return Status;
	}

	Count = *(PULONG)Buffer;

	pModuleInfo = (PSYSTEM_MODULE_INFORMATION_ENTRY)((ULONG)Buffer + 0x04);
	
	for( i = 0; i < Index; i++)
	{
		pModuleInfo++;
	}

	strcat(ModuleName, pModuleInfo->ImageName);
	*ModuleBase = (ULONG)pModuleInfo->Base;

	return STATUS_SUCCESS;
}

VOID
AnsiToUnicode(
	OUT PWCH Unicode,
	IN PCHAR Ansi)
/*++

Routine Description:

	convert ansi string to unicode string

Arugments:

	Unicode - pointer to buffer receive the unicode string. 
	Ansi - pointer to ansi string.

Return Value:

	None

--*/
{
	ULONG i;

	for( i = 0; Ansi[i] != '\0'; i++)
	{
		Unicode[i] = (WCHAR)Ansi[i];
	}
	Unicode[i] = L'\0';
}


ULONG
GetExportData(
	IN PVOID FileBase,
	IN PCHAR DataName)
/*++

Routine Description:

	get data offset from unmapped file.

Arguments:

	FileBase - pointer to buffer filled with file.
	DataName - name of data 

Return Value:

	offset of data.

--*/
{
	ULONG Left;
	ULONG Right;
	ULONG Index;
	ULONG ExpSize;
	LONG RetValue;
	ULONG Temp;
	ULONG MappedExpBase;
	ULONG Name;
	PUSHORT AddressOfNameOrdinals;
	PULONG AddressOfNames;
	PULONG AddressOfFunctions;
	PIMAGE_EXPORT_DIRECTORY Export;


	//
	//	get the offset of export table in unmapped file
	//
	Export = RtlImageDirectoryEntryToData( FileBase,
										   FALSE,
										   IMAGE_DIRECTORY_ENTRY_EXPORT,
										   &ExpSize);
	
	//
	//	get the offset of export table in mapped file
	//
	MappedExpBase = (ULONG)RtlImageDirectoryEntryToData( FileBase,
												  TRUE,
												  IMAGE_DIRECTORY_ENTRY_EXPORT,
												  &ExpSize);
	
	if(Export == NULL || MappedExpBase == 0)
	{
		return 0;
	}


	//
	//	binary search
	//

	Right = Export->NumberOfNames;
	Left = 0;

	AddressOfNameOrdinals = (PUSHORT)(Export->AddressOfNameOrdinals + (ULONG)FileBase - (ULONG)Export + MappedExpBase);
	AddressOfFunctions = (PULONG)(Export->AddressOfFunctions + (ULONG)FileBase - (ULONG)Export + MappedExpBase);
	AddressOfNames = (PULONG)(Export->AddressOfNames + (ULONG)FileBase - (ULONG)Export + MappedExpBase);

	do
	{
		Index = (Right + Left) >> 1;
		Temp = * (AddressOfNameOrdinals + Index);
		Name = *(AddressOfNames + Index) + (ULONG)FileBase - (ULONG)Export + MappedExpBase;

		RetValue = StringCompare( (PCHAR)Name, DataName);
		if(RetValue == 0)
		{
			RetValue = *(AddressOfFunctions + Temp);
			return RetValue;
		}
		else if(RetValue > 0)
		{
			Right = Index - 1;
		}
		else
		{
			Left = Index + 1;
		}
	}while(Left <= Right);

	return 0;
}


LONG 
StringCompare(
	IN PCHAR Str1,
	IN PCHAR Str2)
{
	ULONG i = 0;
	
	do
	{
		if(Str1[i] != Str2[i])
		{
			return Str1[i] - Str2[i];
		}

		if(Str1[i] == '\0')
		{
			return 0;
		}

		i++;
		
	}while(TRUE);
}

ULONG
GetImageBase(
	IN PVOID Buffer)
{
	ULONG Temp;
	ULONG RetValue;

	Temp = (ULONG)Buffer + *(PULONG)((ULONG)Buffer + 0x3c);
	
	RetValue = *(PULONG)(Temp + 0x34);

	return RetValue;
}

ULONG
GetServiceTableAddress(
	IN PVOID Buffer,
	IN ULONG Offset)
/*++

Routine Description:

	get KiServiceTable's address with KeServiceDescriptorTable's
	offset in file.

Arguments:

	Buffer - pointer to buffer filled with nt file.
	Offset - offset of KeServiceDescriptorTable.

Return Value:

	the address of KiServiceTable.
--*/
{
	ULONG RetValue;
	ULONG pReloc;
	ULONG RelocSize;
	ULONG RelocBase;
	ULONG TableSize;
	ULONG Temp;
	ULONG Number;
	ULONG i;

	//
	//	get the offset of relocation table in file 
	//

	pReloc = (ULONG)RtlImageDirectoryEntryToData( Buffer,
										   FALSE,
										   IMAGE_DIRECTORY_ENTRY_BASERELOC,
										   &RelocSize);

	if(pReloc == 0)
	{
		return STATUS_UNSUCCESSFUL;
	}

	//
	//	look over the relocation table to find KiServiceTable
	//

	for(;;)
	{
		RelocBase = *(PULONG)(pReloc);
		TableSize = *(PULONG)(pReloc + 0x04);

		if(TableSize == 0)
		{
			break;
		}

		Number = (TableSize - 0x08) >> 1;

		for( i = 0; i < Number; i++)
		{
			Temp = *(PUSHORT)(pReloc + 0x08 + i * 0x02) & 0x0FFF;
			Temp += RelocBase;
			Temp += (ULONG)Buffer;

			//
			//	C7 05 XX XX XX XX YY YY  YY YY
			//	mov  ds:_KeServiceDescriptorTable, offset _KiServiceTable
			//

			if(*(PULONG)Temp == Offset)
			{
				if(*(PUSHORT)(Temp - 2) == 0x05C7)
				{
					RetValue = *(PULONG)(Temp + 4);
					return RetValue;
				}
			}
		}
		pReloc += TableSize;
	}
	
	return 0;
}