
#include <stdio.h>
#include <string.h>
#include "disarm.h"

static int current_offset = 0;
int color;


/* String constants */
static const char *registers[16] = {
	"r0", "r1", "r2", "r3",
	"r4", "r5", "r6", "r7",
	"r8", "r9", "r10","fp",
	"ip", "sp", "lr", "pc"
};

static const char *cregister[16] = {
	"cr0",  "cr1",  "cr2",  "cr3",
	"cr4",  "cr5",  "cr6",  "cr7",
	"cr8",  "cr9",  "cr10", "cr11",
	"cr12", "cr13", "cr14", "cr15"
};

static const char *stacktext[4][2] = {
	{ "da", "Empty, descending" },
	{ "ia", "Empty, ascending" }, // is this db?? 0xe92dd800 is stmdb
	{ "db", "Full,  descending" },
	{ "ib", "Empty, ascending" }
};

static const char *opcodes[16] = {
	"and", "eor", "sub", "rsb",
	"add", "adc", "sbc", "rsc",
	"tst", "teq", "cmp", "cmn",
	"orr", "mov", "bic", "mvn"
};

static const char *shifts[4] = {
	"lsl", "lsr", "asr", "ror"
};

/* not used ? */
static const char *condtext[16][2] = {
	{ "eq", "Equal (Z)" },
	{ "ne", "Not equal (!Z)" },
	{ "cs", "Unsigned higher or same (C)" },
	{ "cc", "Unsigned lower (!C)" },
	{ "mi", "Negative (N)" },
	{ "pl", "Positive or zero (!N)" },
	{ "vs", "Overflow (V)" },
	{ "nv", "No overflow (!V)" },
	{ "hi", "Unsigned higher (C&!Z)" },
	{ "ls", "Unsigned lower or same (!C|Z)" },
	{ "ge", "Greater or equal ((N&V)|(!N&!V))" },
	{ "lt", "Less than ((N&!V)|(!N&V)" },
	{ "gt", "Greater than(!Z&((N&V)|(!N&!V)))" },
	{ "le", "Less than or equal (Z|(N&!V)|(!N&V))" },
	{ "", "Always" },
	{ "nv", "Never - Use MOV R0,R0 for nop" }
};




/* Working areas */
static char workstr[DISARM_MAXSTRINGSIZE];
static char shiftstr[DISARM_MAXSTRINGSIZE];
static char disasmstr[DISARM_MAXSTRINGSIZE];


/* Functions to decode operands */

static char *shiftrm (QByte  data)
{
	ShiftRm shiftdata ;
    int a =0;
	memcpy(&shiftdata ,&data, sizeof(shiftdata));
	a = sizeof(shiftdata);
	shiftstr[0]=0;
	if (0 == shiftdata.shiftimmed.mustbe0)
	{

     	/* Immediate */
		if (0 == shiftdata.shiftimmed.shiftconst)
		{
			switch (shiftdata.shiftimmed.shifttype)
			{
			case 0: /* LSL */
				sprintf(shiftstr, "%s, lsl #0",
					registers[shiftdata.shiftimmed.Rm]);
				break;
			case 1: /* LSR */
				sprintf(shiftstr, "%s, lsr #32",
					registers[shiftdata.shiftimmed.Rm]);
				break;
			case 2: /* ASR */
				sprintf(shiftstr, "%s, asr #32",
					registers[shiftdata.shiftimmed.Rm]);
				break;
			case 3: /* ROR */
				sprintf(shiftstr, "%s, rrx",
					registers[shiftdata.shiftimmed.Rm]);
				break;
			}
		}
		else
		{

			sprintf(shiftstr, "%s,%s #%d",
				registers[shiftdata.shiftimmed.Rm],
				shifts[shiftdata.shiftimmed.shifttype],
				shiftdata.shiftimmed.shiftconst);
		}
	}
	else
	{
		/* Register */
		sprintf(shiftstr, "%s,%s %s",
			registers[shiftdata.shiftreg.Rm],
			shifts[shiftdata.shiftreg.shifttype],
			registers[shiftdata.shiftreg.Rs]);
	}
	return (shiftstr);
}

static char *cprelative (QByte Rn, QByte offset, QByte U, QByte P)
{
	if (0 == P) {
		sprintf(workstr, "[%s,#%s%2.2lX]",
			cregister[Rn],
			(0 == U)?"-":"",
			offset);
	} else {
		sprintf(workstr, "[%s],#%s%2.2lX",
			cregister[Rn],
			(0 == U)?"-":"",
			offset);
	}
	return (workstr);
}

static char *reglist (QByte reglst)
{
	int mask = 1;
	int reg;
	int pos = 0;
	int inrun = FALSE;

	for (reg=0; reg<16; reg++) {
		if (0 != (reglst & mask)) {
			if (0 != pos)
				workstr[pos++]=(inrun)?'-':',';
			sprintf(workstr+pos, registers[reg]);
			if (inrun == TRUE) pos--;
			else pos = pos + strlen(registers[reg]);

			inrun = TRUE;
		} else
			inrun = FALSE;
		mask = mask << 1;
	}
	return (workstr);
}

static char *dataoperand (QByte operand, QByte I)
{
	QByte data, rotate;
	if (0 == I) {
#if ee
		sprintf(workstr, "%s",
			shiftrm((ShiftRm)operand));
#endif
	} else {
		rotate = (operand >> 8) * 2;
		data = operand & 0xFF;
		sprintf(workstr, " 0x%lX",
			(data >> rotate)|(data<<(32-rotate)));
	}
	return (workstr);
}
//immmediate offset/index
//register offset/index
//scaled register offset/index
static char *sdtsource (QByte Rn, QByte offset, QByte W, QByte U, QByte P, QByte I)
{
	if (0 == I) {
		//immediate
		if (0 == P) {
			sprintf(workstr, "[%s],#%s0x%7.7lX",
					registers[Rn], (0 == U)?"-":"",
					offset);
		} else {
			sprintf(workstr, "[%s,#%s0x%7.7lX]%s",
					registers[Rn], (0 == U)?"-":"",
					offset, (0 == W)?"":"!");
		}

	} else {
          //P ==0  , indicate the use of postindex addressing
		if (0 == P) {
			//register offset/index
			if (0==offset&0xFF0)
			{
			sprintf(workstr, "[%s],#%s %s",
				registers[Rn], (0 == U)?"-":"",
				shiftrm(offset));
			}
			else 
			{
			sprintf(workstr, "[%s],#%s %s",
				registers[Rn], (0 == U)?"-":"",
				shiftrm(offset));
			}
		} 
		//P==1 indicate the use of offset addressing or pre-indexed addressing
		else 
		{
			//register offset/index
			if (0==offset&0xFF0)
			{ 
				sprintf(workstr, "[%s,#%s %s]%s",
				registers[Rn], 
				(0 == U)?"-":"",
				registers[offset&0xF], 
				(0 == W)?"":"!");
			}
			//scaled register offet/index
			else
			{
				sprintf(workstr, "[%s,#%s %s]%s",
				registers[Rn], 
				(0 == U)?"-":"",
				shiftrm(offset), 
				(0 == W)?"":"!");
			}
		}
	}
	return (workstr);
}


/* Decode functions for each instruction type */
static CondPosT decode_swi (SWI swi)
{
	sprintf(disasmstr, "swi%s  0x%08x", 
		   condtext[swi.condition][0],
		swi.comment);
	return (3);
}

static CondPosT decode_data_transfer2 (Bytes bytes)
{
	sprintf(disasmstr, "ldr %s, [%s], 0x%x", 
		registers[bytes.byte1>>4],
		registers[bytes.byte2&0xf],
		(int)(bytes.byte0));
	if ((bytes.byte2>>4) == 2)
		memcpy(disasmstr, "str", 3);
	return (3);
}

static CondPosT decode_data_transfer (Bytes bytes)
{
	sprintf(disasmstr, "ldr %s, [%s, 0x%x]",
		registers[bytes.byte1>>4],
		registers[bytes.byte2&0xf],
		(int)(bytes.byte0));
	if ((bytes.byte2>>4) == 2)
		memcpy(disasmstr, "str", 3);
	return (3);
}

static CondPosT decode_cpregtrans (CPRegTrans cpregtrans)
{
	sprintf(disasmstr, "m%s  p%d,%d,%s,%s,%s,%d",
		(0 == cpregtrans.L)?"cr":"rc",
		cpregtrans.CPNum,
		cpregtrans.CPOpc,
		registers[cpregtrans.Rd],
		cregister[cpregtrans.CRn],
		cregister[cpregtrans.CRm],
		cpregtrans.CP);
	return (3);
}


static CondPosT decode_cpdataop (CPDataOp cpdataop)
{
	sprintf(disasmstr, "cdp  p%d,%d,%s,%s,%s,%d",
		cpdataop.CPNum,
		cpdataop.CPOpc,
		cregister[cpdataop.CRd],
		cregister[cpdataop.CRn],
		cregister[cpdataop.CRm],
		cpdataop.CP);
	return (3);
}


static CondPosT decode_cpdatatrans (CPDataTrans cpdatatrans)
{
	sprintf(disasmstr, "%sc  p%d,N%d%s,%s%s",
		(0 == cpdatatrans.L)?"st":"ld",
		cpdatatrans.CPNum,
		cpdatatrans.N,
		cregister[cpdatatrans.CRd],
		cprelative(cpdatatrans.Rn,
		cpdatatrans.offset,
		cpdatatrans.U,
		cpdatatrans.P),
		(0==cpdatatrans.W)?"":"!");

	if (0!=cpdatatrans.L) {
		if (0x0E != cpdatatrans.condition) 
			disasmstr[5]='l';
		else
			disasmstr[3]='l';
	}
	return (3);
}


static CondPosT decode_branch (Branch branch)
{
	QByte cp;
	char label[512];
	int addr = 0;
	unsigned char *ptr = (unsigned char *)&addr;
	Bytes* bytes = (Bytes *)&branch;

	ptr[2] = bytes->byte2;
	ptr[1] = bytes->byte1;
	ptr[0] = bytes->byte0;
	if (branch.sign)
		ptr[3]=0xff;
	addr*=4;
	addr+=8;

	label[0]='\0';
	string_flag_offset(label, current_offset+addr);
	sprintf(disasmstr, "b%s  0x%x ; %s%s",
		
		condtext[(bytes->byte3&0xf0)>>4][0],
		(0==branch.sign)?(current_offset+addr):
		(current_offset+(addr)), 
		label,
		(color)?"\e[0m":"");
	if (0 == branch.L) {
		cp = 1;
	} else {
		disasmstr[(color)?6:1]='l';
		cp = 2;
	}
	return (cp);
}




static CondPosT decode_undefined (Access32 instruction)
{
	sprintf(disasmstr, "dcd  0x%lx", instruction.qbyte);
	return (0);
}


/*
LDR|STR {<cond>}{B}{T}   Rd,<addressing_mode>
typedef struct singledatatrans {
	QByte offset : 12;
	QByte Rd : 4;
	QByte Rn : 4;
	QByte L : 1;
	QByte W : 1;
	QByte B : 1;
	QByte U : 1;
	QByte P : 1;
	QByte I : 1;
	QByte mustbe01 : 2;
	QByte condition : 4;
} SingleDataTrans;
L bit distinguish between ad load(L==1) and a Store instruction L==0)
B bit Distinguishes between an unsigned byte(B==1) and a word(B==0) access
There are nine addressing modes used to calculate the address for a Load and Store word or Unsigned Byte instruction.
All nine of the following options are available for LDR, LDRB,STR nd STRB, For LDRBT, LDRT, STRBT, and STRBT,only the post-indexed options
(the last threein the list ) are availabe.
For The PLD instruction ,only the offset options (the first three in the list ) are available.
[<Rn>,#+/-<offset_12>]
[<Rn>,+/-<Rm>]
[<Rn>,+/-<Rm>, <shift> #<shift_imm>]
[<Rn>,#+/-<offset_12>]!
[<Rn>,+/-<Rm>]!
[<Rn>,+/-<Rm?,<shift> #shift_imm>]!
[<Rn>],#+/-<offset_12>
[<Rn>],+/-<Rm>
[<Rn>],+/-<Rm>, <shift>#<shift_imm>
*/
static CondPosT decode_singledatatrans (SingleDataTrans singledatatrans)
{
	QByte cp;

	
	/* Single Data Transfer */
	sprintf(disasmstr, "%sr%s    %s,%s",
		(0 == singledatatrans.L)?"st":"ld",
		condtext[singledatatrans.condition][0],
		registers[singledatatrans.Rd],
		sdtsource(singledatatrans.Rn,
		singledatatrans.offset, /* Offset field */
		singledatatrans.W, /* Writeback */
		singledatatrans.U, /* up/down */
		singledatatrans.P, /* Pre/post indexed */
		singledatatrans.I)); /* Immediate */
	cp = 11 + ((0x0E != singledatatrans.condition)?2:0);
	/* Byte (8bit) or Word (32bit) transfer */
	if (0 != singledatatrans.B) disasmstr[cp++]='B';
	/* T is present if writeback and post-indexed */
	if ((0 == singledatatrans.P) &&
		(0 != singledatatrans.W)) disasmstr[cp++]='T';
	return (3);
}

//LDR|STR {<cond>} H|SH|SB|D <Rd>, <addressing_mode>
//1.[<Rn>, #+/-<offset_8>]
//2.[<Rn>, +/-<Rm>]
//3.[<Rn>, #+/-<offset_8>]!
//4.[<Rn>, +/-<Rm>]!
//5.[<Rn>],#+/-<offset_8>
//6.[<Rn>], +/-<Rm>

static CondPosT decode_misceldatatrans (SingleDataTrans singledatatrans)
{
	QByte cp;

		
	/* Single Data Transfer */
	sprintf(disasmstr, "%sr%s    %s,%s",
		(0 == singledatatrans.L)?"st":"ld",
		condtext[singledatatrans.condition][0],
		registers[singledatatrans.Rd],
		sdtsource(singledatatrans.Rn,
		singledatatrans.offset, /* Offset field */
		singledatatrans.W, /* Writeback */
		singledatatrans.U, /* up/down */
		singledatatrans.P, /* Pre/post indexed */
		singledatatrans.I)); /* Immediate */
	cp = 11 + ((0x0E != singledatatrans.condition)?2:0);
	/* Byte (8bit) or Word (32bit) transfer */


	//?????
	///* T is present if writeback and post-indexed */
	//if ((0 == singledatatrans.P) &&
	//	(0 != singledatatrans.W)) disasmstr[cp++]='T';
	return (3);
}

//swp is ok
static CondPosT decode_singledataswap (SingleDataSwap singledataswap)
{
	sprintf(disasmstr, "swp%s  %s,%s,[%s]",
		condtext[singledataswap.condition][0],
		registers[singledataswap.Rd],
		registers[singledataswap.Rm],
		registers[singledataswap.Rn]);
	if (0 != singledataswap.B) disasmstr[3 + ((0x0E != singledataswap.condition)?2:0)]='B';
	return (3);
}

/*
Normal Multiply
MUL MLA

Long multiply

*/
static CondPosT decode_multiply (Multiply multiply)
{
	if ( 0==multiply.LongMul)
	{
		if (0 == multiply.A) {
			//Cond 0000000 S Rd SBZ Rs 1001 Rm
			//MUL{<cond>}{S} <Rd>, <Rm>, <Rs>
			sprintf(disasmstr, "mul%s  %s, %s, %s",
				condtext[multiply.condition][0],
				registers[multiply.Rd],
				registers[multiply.Rm],
				registers[multiply.Rs]);
		} else {
			//Cond 0000001 S <Rd>,<Rn>,<Rs>,1001,Rm
			//MLA{<cond>}{S} <Rd>, <Rm>, <Rs>, <Rn>
			sprintf(disasmstr, "mla%s  %s, %s, %s, %s",
				condtext[multiply.condition][0],
				registers[multiply.Rd],
				registers[multiply.Rm],
				registers[multiply.Rs],
				registers[multiply.Rn]);
		}
		if (0 != multiply.S) disasmstr[3 + ((0x0E != multiply.condition)?2:0)]='S';
	}
	else
	{
		if (0 == multiply.A) {
			//Cond 0000000 S Rd SBZ Rs 1001 Rm
			//MUL{<cond>}{S} <Rd>, <Rm>, <Rs>
			sprintf(disasmstr, "smull%s  %s, %s, %s",
				condtext[multiply.condition][0],
				registers[multiply.Rd],
				registers[multiply.Rn],
				registers[multiply.Rm],
				registers[multiply.Rs]);
		} else {
			//Cond 0000001 S <Rd>,<Rn>,<Rs>,1001,Rm
			//MLA{<cond>}{S} <Rd>, <Rm>, <Rs>, <Rn>
			sprintf(disasmstr, "smlal%s  %s, %s, %s, %s",
				condtext[multiply.condition][0],
				registers[multiply.Rd],
				registers[multiply.Rn],
				registers[multiply.Rm],
				registers[multiply.Rs]);
		}
		if (0 != multiply.S) disasmstr[5 + ((0x0E != multiply.condition)?2:0)]='S';
	}
	return (3);
}


// command clz
CondPosT decode_clz (CmdCLZ clz)
{
	sprintf(disasmstr, "clz%s %s, %s", 
		condtext[clz.cond][0],
		   registers[clz.Rd],
		   registers[clz.Rm]);
	return 3;
}

CondPosT decode_branchExtend(CmdBranchExtend branch)
{
	if ( 1==branch.bxblxdif)
	{
	sprintf(disasmstr, "bx%s %s", 
		condtext[branch.cond][0],
		   registers[branch.Rm]);

	}
	else if (3==branch.bxblxdif)
	{
	sprintf(disasmstr, "blx%s %s", 
		condtext[branch.cond][0],
		   registers[branch.Rm]);

	}
	return 3;
}

/*
	typedef struct miscelone {
	QByte dontcareA : 4;
	QByte mustbe1	: 1;
	QByte dontcareB : 2;
	QByte mustbe0L  : 1;
	QByte dontcareC : 12;
	QByte mustbe0H	: 1;
	QByte dontcareD : 2;
	QByte mustbe10	: 2;
	QByte mustbe000 : 3;
	QByte condition : 4;
} MiscellanesInstructOne;
*/
static CondPosT decode_miscelinstrone(MiscellanesInstructOne miscInstr)
{
	CondPosT condpos;
	
	if ( miscInstr.mustbe1 )
	{
		//software breakpoint[2,3]
		if  (0==miscInstr.dontcareB)
		{
			if ( ( 1==miscInstr.dontcareD )
				&& (0xFFF==miscInstr.dontcareC)  //cond 00010 010 SBO SBO SBO 0001 Rm
				)
			{//branch/exchange instruction set[1]
			    condpos = decode_branchExtend(*(CmdBranchExtend*) &miscInstr);
			}
			else if (3==miscInstr.dontcareD)
			{//count leading zeros
				condpos = decode_clz( *( CmdCLZ* )&miscInstr );
			}
		}
		else if (1==miscInstr.dontcareB)
		{
			if ( ( 1==miscInstr.dontcareD)
				&& ( 0xFFF==miscInstr.dontcareC)//cond 00010 010 SBO SBO SBO 0011 Rm
				)
			{//branch and link/exchange instruction set[2]
				condpos = decode_branchExtend(*(CmdBranchExtend*) &miscInstr);
			}
		}
		else if (2==miscInstr.dontcareB)
		{//Enhanced DSP add/subtracts[4]

		}
		else if (3==miscInstr.dontcareB)
		{
			if (1==miscInstr.dontcareD )
			{//software breakpoint[2,3]
				//?? bkpt
				//decode_swi(*((SWI*)&miscInstr));
			}
		}

	}
	else
	{
		//Enhanced DSP multiplies[4]
		if  (1==miscInstr.mustbe0L)
		{

		}
		
		
		else
		{ 

			//move register to status register
			if ( 1== (miscInstr.dontcareD&1) 
				&& 0xF0== (miscInstr.dontcareC&0xFF))
			{//MSR
			//Here we need make sure that we need take care of other condition.
				//I will do this in later time.
				decode_msrregtopsr ( *(MSRRegToPSR*) &miscInstr);
			}
			else
			{  //move status register to register
				//MRS
				MRSPSRToReg *pmrsprstoreg =  (MRSPSRToReg*)& miscInstr;
				if ( (0==pmrsprstoreg->mustbe000000000000)
					&& (2==pmrsprstoreg->mustbe00010)
					&& (0xf==pmrsprstoreg->mustbe001111)
					)
				{  //make sure 
					decode_mrspsrtoreg ( *pmrsprstoreg) ;
				}
			}
		}

		//decode_msrregtopsr (MSRRegToPSR msrregtopsr)

	}
	return 3;
}
//typedef struct multiplyLoadStore {
//	QByte Rm : 16;
//	QByte Rn : 4;
//	QByte L : 1;
//	QByte W : 1;
//	QByte S : 1;
//	QByte U : 1;
//	QByte P : 1;
//	QByte mustbe100 : 3;
//	QByte condition : 4;
//} LoadStoreMultiply;
//LDM { <cond> } <addressing_mode> Rn {!}, <registers> {^}
//STM { <cond> } <addressing_mode> Rn {!}, <registers> {^}
static CondPosT  decode_loadstoremultiple(LoadStoreMultiple loadstorem)
{  int a= 0;  
QByte storetype;

storetype = loadstorem.P * 2 + loadstorem.U;

sprintf(disasmstr, "%sm%s     %s%s,{%s}%s",
		(0 == loadstorem.L)?"st":"ld",
		condtext[loadstorem.condition][0],
		registers[loadstorem.Rn],
		(0 == loadstorem.W)?"":"!", /* writeback */
		reglist( loadstorem.registerlist ),
		(0 == loadstorem.S)?"":"^"); /* S-bit, load CPSR */

strncpy(disasmstr+3+((0x0E != loadstorem.condition)?2:0),
		stacktext[storetype][0], 2);

return (3);
}





static CondPosT decode_msrtopsrflags (MSRToPSRFlags msrtopsrflags)
{
	QByte data, rotate;

	if (0 == msrtopsrflags.I) {
		sprintf(disasmstr, "msr  %sPSR_flg,%s",
			(0 == msrtopsrflags.Pd)?"c":"s",
			registers[msrtopsrflags.operand & 0xF]);
	} else {


		data = msrtopsrflags.operand & 0xFF;
		rotate = msrtopsrflags.operand >> 8;
		sprintf(disasmstr, "msr  %sPSR_flg,#%8.8lX",
			(0 == msrtopsrflags.Pd)?"c":"s",
			(data >> rotate) | (data << (32 - rotate)));
	}
	return (3);
}


static CondPosT decode_mrspsrtoreg (MRSPSRToReg mrspsrtoreg)
{
	sprintf(disasmstr, "mrs%s  %s,%sPSR",
		condtext[mrspsrtoreg.condition][0],
		registers[mrspsrtoreg.Rd],
		(0 == mrspsrtoreg.Ps)?"c":"s");
	return (3);
}

static CondPosT decode_msrimetoreg (MSRImeToSReg mrspsrtoreg)
{
	sprintf(disasmstr, "msr%s  %sPSR_%s,#%x",
		condtext[mrspsrtoreg.condition][0],//CONDITION
		(0 == mrspsrtoreg.R)?"c":"s",
		0,//field
		mrspsrtoreg.immediate);
	return (3);
}

static CondPosT decode_msrregtopsr (MSRRegToPSR msrregtopsr)
{
//	int a = sizeof(MSRRegToPSR);
//	unsigned int mm;
	//memcpy(&mm,&msrregtopsr, sizeof (mm));
	sprintf(disasmstr, "msr%s  %spsr_%s,%s",
		condtext[msrregtopsr.condition][0],
		(0 == msrregtopsr.R)?"c":"s",
		0,
		registers[msrregtopsr.Rm]);
	return (3);
}





//Բshiftķʽٷ֧ʱı
//Է
static CondPosT decode_dataproc (DataProc dataproc)
{
	Bytes *bytes = (Bytes *)&dataproc;
	char pfx[16];

	pfx[0]='\0';
	if (
		bytes->byte0==0
		&& bytes->byte1 == 0
		&& bytes->byte2 == 0
		&& bytes->byte3 == 0) {
			//strcpy(pfx,"\e[36m");
	}
	// <opcode1>{<cond>}{S} <Rd>,<shifter_operand>
	// <opcode1> := MOV|MVN
	// <opcode2>{<cond>} <Rn>,<shifter_operand>
	// <opcode2 := CMP | CMN | TST | TEQ
	// <opcode3> { <cond> {S} <Rd>, <Rn>, <shifter_operand>
	//  <opcode3> := ADD | SUB | RSB | ADC |SBC | RSC |AND |BIC |EOR | ORR
	//I bit Distinguishes between the immediate and register forms of <shifter_operand>


	// <opcode1>{<cond>}{S} <Rd>,<shifter_operand>
	// <opcode1> := MOV|MVN
	//  MOV=1101,MVN=1111
	if ( (dataproc.opcode == MOV)  || (dataproc.opcode == MVN)) { 
		if (dataproc.I) { /*immediate data */
			/* MOV/MVN */
			sprintf(disasmstr, "%s%s%s%s %s, %s",
				pfx, opcodes[dataproc.opcode],
				condtext[dataproc.condition][0],
				((dataproc.S) ?"s":""),
				registers[dataproc.Rd], 
				dataoperand(dataproc.operand2, dataproc.I));
		} else {
			/* MOV between registers*/
			if (dataproc.Rd > 15 ) //MOVָЩ޶
				sprintf(disasmstr, "Invalid opcode");
			else 
			{

				if ( 0==(dataproc.operand2&0xFF0) ) //5.1.4  <Rm> Specifies the register whose value is the instruction operand
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F]);
				}
				else if ( 0x60 == (dataproc.operand2 &0xFF0) ) //5.1.13 Rotate right with extend
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, rrx",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F]);

				}
				else if ( 0x70==(dataproc.operand2 &0x0F0) )//5.1.12 //rm ,ror rs //tested
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, ror %s",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						registers[(dataproc.operand2&0xF00)>>8]
					);
				}
				else if ( 0x10 ==(dataproc.operand2 & 0x0F0) ) //5.1.6 logical shift left by register
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, lsl %s",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						registers[(dataproc.operand2&0xF00)>>8]
					);

				}
				else if ( 0x30==(dataproc.operand2 &0x0F0) ) //5.1.8 Logical shift right by regsiter
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, lsr %s",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						registers[(dataproc.operand2&0xF00)>>8]
					);
				}
				else if ( 0x50==(dataproc.operand2 &0x0F0) )//5.1.10  //Arithmetic shift right by register
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, asr %s",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						registers[(dataproc.operand2&0xF00)>>8]
					);
				}
				else if ( 0x0 ==(dataproc.operand2 &0x070) ) //5.1.5 logical shift left by immediate
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, lsl #%d",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						(dataproc.operand2&0xF80)>>7
					);
				}
				else if ( 0x20==(dataproc.operand2 &0x070) )//5.1.7  //logical shift right by immediate
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, lsr #%d",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						(dataproc.operand2&0xF80)>>7
					);
				}
				else if ( 0x40==(dataproc.operand2 &0x070) ) //5.1.9 Arithmetic shift right by immediate
				{
					sprintf(disasmstr, "%s%s%s%s %s, %s, asr #%d",
						pfx, opcodes[dataproc.opcode],
						condtext[dataproc.condition][0],
						((dataproc.S) ?"s":""),
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						(dataproc.operand2&0xF80)>>7
					);
				}
				else if ( 0x60==(dataproc.operand2 &0x070) )//5.1.11 Rotate right by immediate
				{
					sprintf(disasmstr, "%s%s%s  %s, %s, %s, ror #%d",
						pfx, opcodes[dataproc.opcode],
						condtext[(bytes->byte3&0xf0)>>4][0],
						registers[dataproc.Rd],
						registers[dataproc.operand2&0x0F],
						(dataproc.operand2&0xF80)>>7
						);
				}
				else
				{

				}
			}  
		}
	} 
	// <opcode2>{<cond>} <Rn>,<shifter_operand>
	// <opcode2 := CMP | CMN | TST | TEQ
	//here we can see there is no sign 
	else 	if ((dataproc.opcode >= TST) && (dataproc.opcode <= CMN))
	{

		if (dataproc.I)  /*immediate data */
		{
			sprintf(disasmstr, "%s%s%s %s, %s",
				pfx, opcodes[dataproc.opcode],
				condtext[dataproc.condition][0],
				registers[dataproc.Rn], 
				dataoperand(dataproc.operand2, dataproc.I));
		} 
		else 
		{

			if ( 0==(dataproc.operand2 &0xFF0) ) //5.1.4  <Rm> Specifies the register whose value is the instruction operand
			{
				sprintf(disasmstr, "%s%s%s %s, %s",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F]);
			}
			else if ( 0x60 == (dataproc.operand2 &0xFF0) ) //5.1.13 Rotate right with extend
			{
				sprintf(disasmstr, "%s%s%s %s, %s, rrx",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F]);

			}
			else if ( 0x70==(dataproc.operand2 &0x0F0) )//5.1.12 //rm ,ror rs //tested
			{
				sprintf(disasmstr, "%s%s%s %s, %s, ror %s",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);
			}
			else if ( 0x10 ==(dataproc.operand2 & 0x0F0) )//5.1.6 logical shift left by register
			{
				sprintf(disasmstr, "%s%s%s %s, %s, lsl %s",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);

			}
			else if ( 0x30==(dataproc.operand2 &0x0F0) ) //5.1.8 Logical shift right by regsiter
			{
				sprintf(disasmstr, "%s%s%s %s, %s, lsr %s",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);
			}
			else if ( 0x50==(dataproc.operand2 &0x0F0) )//5.1.10  //Arithmetic shift right by register
			{
				sprintf(disasmstr, "%s%s%s %s, %s, asr %s",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);
			}
			else if ( 0x0 ==(dataproc.operand2 &0x070) ) //5.1.5 logical shift left by immediate
			{
				sprintf(disasmstr, "%s%s%s %s, %s, lsl #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
				);
			}
			else if ( 0x20==(dataproc.operand2 &0x070) )//5.1.7  //logical shift right by immediate
			{
				sprintf(disasmstr, "%s%s%s %s, %s, lsr #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
				);
			}
			else if ( 0x40==(dataproc.operand2 &0x070) ) //5.1.9 Arithmetic shift right by immediate
			{
				sprintf(disasmstr, "%s%s%s %s, %s, asr #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
				);
			}
			else if ( 0x60==(dataproc.operand2 &0x070) )//5.1.11 Rotate right by immediate
			{
				sprintf(disasmstr, "%s%s%s %s, %s, ror #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
					);
			}
			else
			{

			}

		}

	}
	// <opcode3> { <cond> {S} <Rd>, <Rn>, <shifter_operand>
	//  <opcode3> := ADD | SUB | RSB | ADC |SBC | RSC |AND |BIC |EOR | ORR
	else
	{ 
		//immediate data
		if (dataproc.I) 
		{
			/* AND/EOR/SUB/RSB/ADC/SBC/RSC/ORR/BIC */
			sprintf(disasmstr, "%s%s  %s, %s, %s",
				pfx, opcodes[dataproc.opcode],
				registers[dataproc.Rd],
				registers[dataproc.Rn],
				dataoperand(dataproc.operand2, dataproc.I));
		} 
		/* ADD/ADC/SBC/RSC/ORR/BIC */  //here we need big change. For we have some condition to take care of.//songhao
		else 	
		{
			if ( 0==(dataproc.operand2 &0xFF0) ) //5.1.4  <Rm> Specifies the register whose value is the instruction operand
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F]);
			}
			else if ( 0x60 == (dataproc.operand2 &0xFF0) ) //5.1.13 Rotate right with extend
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, rrx",
					pfx, opcodes[dataproc.opcode],
					condtext[dataproc.condition][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F]
				);
			}
			else if ( 0x70==(dataproc.operand2 &0x0F0) )//5.1.12 //rm ,ror rs //tested
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, ror %s",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);

			}
			else if ( 0x10 ==(dataproc.operand2 & 0x0F0) ) //5.1.6 logical shift left by register
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, lsl %s",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);

			}
			else if ( 0x30==(dataproc.operand2 &0x0F0) ) //5.1.8 Logical shift right by regsiter
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, lsr %s",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);
			}
			else if ( 0x50==(dataproc.operand2 &0x0F0) )//5.1.10  //Arithmetic shift right by register
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, asr %s",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					registers[(dataproc.operand2&0xF00)>>8]
				);

			}
			else if ( 0x0 ==(dataproc.operand2 &0x070) ) //5.1.5 logical shift left by immediate
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, lsl #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
					);
			}
			else if ( 0x20==(dataproc.operand2 &0x070) )//5.1.7  //logical shift right by immediate
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, lsr #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
					);
			}
			else if ( 0x40==(dataproc.operand2 &0x070) ) //5.1.9 Arithmetic shift right by immediate
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, asr #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
					);
			}
			else if ( 0x60==(dataproc.operand2 &0x070) )//5.1.11 Rotate right by immediate
			{
				sprintf(disasmstr, "%s%s%s%s  %s, %s, %s, ror #%d",
					pfx, opcodes[dataproc.opcode],
					condtext[(bytes->byte3&0xf0)>>4][0],
					((dataproc.S) ?"s":""),
					registers[dataproc.Rd],
					registers[dataproc.Rn],
					registers[dataproc.operand2&0x0F],
					(dataproc.operand2&0xF80)>>7
					);
			}
			else 
			{
			}
		}
	}

	//if (0 != dataproc.S) disasmstr[3 + ((0x0E != dataproc.condition)?2:0)]='S';
	return (3);
}


static CondPosT decode_unknown (Access32 instruction)
{
	sprintf(disasmstr, "; data (0x%08x)", (unsigned int)instruction.qbyte);
	return (0);
}

char *disarm2 (RawInstruction rawinstruction, int offset)
{
	CondPosT condpos;
	Access32 instruction;
   int a = sizeof(rawinstruction);
	current_offset = offset;

	//instruction = (Access32)rawinstruction;
	memcpy(&instruction, &rawinstruction,sizeof(rawinstruction));

	sprintf(disasmstr, "Not decoded");
	condpos = 0; /* Position in disasmstr for condition - set to 0 to inhibit insertion of condition */

	if (0xF == instruction.swi.mustbe1111) {
		/* SWI */
		condpos = decode_swi (instruction.swi);
	} 
	else if ( 0xE == instruction.cpregtrans.mustbe1110 ) // Coprocessor registesr transfers/data processing
	{
		if (0x1 == instruction.cpregtrans.mustbe1) {				//coprocessor register transfers 
			/* Coprocessor Register Transfer */
			condpos = decode_cpregtrans (instruction.cpregtrans);
		} else {      
			/* Coprocessor Data Operation */
			condpos = decode_cpdataop (instruction.cpdataop);		//coprocessor data processing
		}

	}
	else if ( 0x6 == instruction.cpdatatrans.mustbe110  ) // Coprocessor load/store and double register transfers[6]
	{
		/* Coprocessor Data Transfer */
		condpos = decode_cpdatatrans (instruction.cpdatatrans);
	}
	else if ( 0x5 == instruction.branch.mustbe101 ) //Branch and branch with link and change to Thumb //Branch and branch with link
	{
		condpos = decode_branch (instruction.branch);
	}
	else if (  (0xF==instruction.undefined4.condition1111) 
		&&  ( 0x4==instruction.undefined4.mustbe100)   ) //Undefined Ins[4]
	{
		condpos = decode_unknown (instruction);
	}
	else if (0x4==instruction.loadstoremultiple.mustbe100) //load/store multiple
	{
		/* load/store mutiple */
		condpos = decode_loadstoremultiple (instruction.loadstoremultiple);

	}
	else if (  (0xF==instruction.undefined47.condition1111) //Undefined Instruction[4,7]
		&& (1==instruction.undefined47.mustbe1) ) 
	{
		condpos = decode_unknown (instruction);
	}
	else if ( (0x3==instruction.undefined.mustbe011)
		&& (1==instruction.undefined.mustbe1) ) //undefined Instruction
	{
		condpos = decode_unknown (instruction);
	}
	else if (0x1==instruction.singledatatrans.mustbe01) //Load/Store register offset Load Store Immediate offset
	{
		/* Single Data Transfer */
		condpos = decode_singledatatrans (instruction.singledatatrans);

	}
	else if ( ( 6==instruction.msrimetosreg.mustbe00110) //Move immediate to status register
		&& (2==instruction.msrimetosreg.mustbe10)
		&& (0==instruction.msrimetosreg.SBO)
		)
	{
		condpos = decode_msrimetoreg(instruction.msrimetosreg);
	}
	else if ( (0==instruction.undefined3.mustbe00) //undefined instruction[3]
		&& (1==instruction.undefined3.mustbe001)
		&& (2==instruction.undefined3.mustbe10) )
	{
		condpos = decode_unknown (instruction);

	}
	else if(1==instruction.dataprocimme.mustbe001) //Data Processing immediate[2]
	{
		condpos = decode_dataproc (instruction.dataproc);

	}
	else if ( (0==instruction.multiextraloadstore.mustbe000) //Multipies ,extra load/stores 
		&& (1==instruction.multiextraloadstore.mustbeH1)
		&& (1==instruction.multiextraloadstore.mustbeL1)
		)
	{
		if ( 0 ==instruction.multiextraloadstore.dontcareB )
		{
			//Muptiply (accumulate
			//multiply long
			//Swap/swap byte
			//decode_multiply
			if ( 1==(instruction.multiextraloadstore.dontcareC & 0x10000))
			{
				SingleDataSwap *psingledataswap = (SingleDataSwap*)&instruction;
				if ( ( 0==psingledataswap->mustbe00)
					&&(9==psingledataswap->mustbe00001001)
					&& (2==psingledataswap->mustbe00010)
					)
				{
					decode_singledataswap ( *psingledataswap );
				}
			}
			else
			{
				//multiply
				Multiply*  pmultiply = (Multiply*)&instruction;
				if( (0==pmultiply->mustbe0000)
					&&(9==pmultiply->mustbe1001)
					)
				decode_multiply ( *pmultiply );
			}
		}
		else
		{//extra load store 
			condpos = decode_singledatatrans (instruction.singledatatrans);
		}
		/* Single Data Transfer */
		
	}
	else if 
		( ( 0== instruction.miscelinstrone.mustbe000) ///miscellaneous insturction
		&&(0==instruction.miscelinstrone.mustbe0H)
		&&(0==instruction.miscelinstrone.mustbe0L)
		&&(2==instruction.miscelinstrone.mustbe10)
		&&(1==instruction.miscelinstrone.mustbe1)
		)
	{
     condpos = decode_miscelinstrone( instruction.miscelinstrone );
	}
	else if ( (0==instruction.dataprocregshift.mustbe000) //data processing register shift[2]
		&& (0==instruction.dataprocregshift.mustbe0)
		&& (1==instruction.dataprocregshift.mustbe1)
		)
	{
		condpos = decode_dataproc (instruction.dataproc);

	}
	else if( (0==instruction.miscelinstrtwo.mustbe000)//miscellaneous instructions;
		&& (2==instruction.miscelinstrtwo.mustbe10)
		&&(0==instruction.miscelinstrtwo.mustbe0H)
		&& (0==instruction.miscelinstrtwo.mustbe0L)
		)
	{
		condpos = decode_miscelinstrone( instruction.miscelinstrone );
	}
	else if  ( (0==instruction.dataprocimmeshit.mustbe000) //data processing immediate shift
		&& (0==instruction.dataprocimmeshit.mustbe0 )
		)
	{
		condpos = decode_dataproc (instruction.dataproc);
	}

	return (disasmstr);
}


int string_flag_offset(char *buf, unsigned long long seek)
{
	return 0;
}